﻿{$reference System.Numerics.dll}
uses System, System.Collections.Generic, System.Linq;

type
  Expression = class
    public 
      Value: real;               // Числовое значение
      ExprStr: string;          // Строковое представление
      UsedFours: integer;       // Сколько четвёрок использовано
      Complexity: integer;      // Сложность выражения
      
      constructor(v: real; e: string; u: integer; c: integer := 0);
      begin
        Value := v;
        ExprStr := e;
        UsedFours := u;
        Complexity := c;
      end;
      
      function IsValid: boolean;
      begin
        // Проверка на NaN и допустимость значения
        if Value > real.MaxValue then
          Result := false
        else if real.IsNaN(Value) then
          Result := false
        else
          Result := true;
      end;
      
      function ToString: string; override;
      begin
        Result := $'{ExprStr} = {Value:F2} (четвёрок: {UsedFours})';
      end;
  end;

type
  FourFoursSolver = class
    private
      MaxInteger: integer;
      AllowedOps: List<char>;
      AllowedUnary: List<string>;
      RequiredFours: integer;
      
      // Результаты с РОВНО 4 четвёрками
      ResultsExactFours: Dictionary<integer, Expression>;
      AllExpressionsExact: List<Expression>;
      
      // Статистика
      Stats: record
        ExpressionsGenerated: integer;
        ExpressionsWith4Fours: integer;
        NumbersFound: integer;
      end;
      
    public
      constructor(maxInt: integer := 100);
      begin
        MaxInteger := maxInt;
        RequiredFours := 4;
        //Stats := (0, 0, 0);
        ResultsExactFours := new Dictionary<integer, Expression>;
        AllExpressionsExact := new List<Expression>;
      end;
      
      procedure ConfigureOperations(opsSet: string);
      begin
        AllowedOps := new List<char>;
        AllowedUnary := new List<string>;
        
        case opsSet of
          'basic':
            begin
              var b: array of char;
              b:= ['+', '-', '*', '/'];
                 AllowedOps.AddRange(b);
              // AllowedUnary остается пустым
            end;
          'intermediate':
            begin
              AllowedOps.AddRange(['+', '-', '*', '/']);
              AllowedUnary.AddRange(['!', 'sqrt', '.']);
            end;
          'advanced':
            begin
              AllowedOps.AddRange(['+', '-', '*', '/']);
              AllowedUnary.AddRange(['!', 'sqrt', '.', '~', 'floor', 'ceil']);
            end;
          'extreme':
            begin
              AllowedOps.AddRange(['+', '-', '*', '/']);
              AllowedUnary.AddRange(['!', 'sqrt', '.', '~', 'floor', 'ceil', '^']);
            end;
          else
            begin
              // По умолчанию intermediate
              AllowedOps.AddRange(['+', '-', '*', '/']);
              AllowedUnary.AddRange(['!', 'sqrt', '.']);
            end;
        end;
      end;
      
      function ApplyUnaryOperation(expr: Expression; op: string): Expression;
      begin
        if not expr.IsValid then Exit(nil);
        
        // Унарные операции НЕ меняют количество четвёрок!
        var currentFours := expr.UsedFours;
        
        case op of
          '!':  // Факториал
            begin
              if (expr.Value >= 0) and (expr.Value <= 15) and 
                 (Abs(expr.Value - Round(expr.Value)) < 1e-10) then
              begin
                var val := integer(Round(expr.Value));
                var res := 1.0;
                for var i := 2 to val do
                  res *= i;
                
                var exprStr := if (expr.ExprStr.Contains('+') or expr.ExprStr.Contains('-')) then
                                $'({expr.ExprStr})!'
                              else
                                $'{expr.ExprStr}!';
                
                Result := new Expression(res, exprStr, currentFours, expr.Complexity + 10);
              end;
            end;
            
          'sqrt':  // Квадратный корень
            begin
              if expr.Value >= 0 then
              begin
                var res := Sqrt(expr.Value);
                var exprStr := if (expr.ExprStr.Contains('+') or expr.ExprStr.Contains('-')) then
                                $'sqrt({expr.ExprStr})'
                              else
                                $'sqrt{expr.ExprStr}';
                
                Result := new Expression(res, exprStr, currentFours, expr.Complexity + 5);
              end;
            end;
            
          '.':  // Десятичная точка
            begin
              if expr.ExprStr = '4' then
                Result := new Expression(0.4, '.4', currentFours, expr.Complexity + 1);
            end;
            
          '~':  // Периодическая дробь
            begin
              if expr.ExprStr = '4' then
                Result := new Expression(4.0 / 9.0, '.4~', currentFours, expr.Complexity + 2);
            end;
            
          'floor':  // Округление вниз
            begin
              if expr.Value > int32.MaxValue then expr.Value := int32.MaxValue;
              if expr.Value < int32.MinValue then expr.Value := int32.MinValue;              
              var res := Floor(expr.Value);
              Result := new Expression(res, $'floor({expr.ExprStr})', currentFours, expr.Complexity + 3);
            end;
            
          'ceil':  // Округление вверх
            begin
              var res := Ceil(expr.Value);
              Result := new Expression(res, $'ceil({expr.ExprStr})', currentFours, expr.Complexity + 3);
            end;
            
          '^':  // Квадрат
            begin
              var res := expr.Value * expr.Value;
              Result := new Expression(res, $'({expr.ExprStr})^2', currentFours, expr.Complexity + 3);
            end;
        end;
      end;
      
      function ApplyBinaryOperation(expr1, expr2: Expression; op: string): Expression;
      begin
        if not expr1.IsValid or not expr2.IsValid then Exit(nil);
        
        var totalFours := expr1.UsedFours + expr2.UsedFours;
        if totalFours > RequiredFours then Exit(nil);
        
        try
          var res: real := 0;
          var exprStr := '';
          var complexity := expr1.Complexity + expr2.Complexity;
          
          case op of
            '+':
              begin
                res := expr1.Value + expr2.Value;
                exprStr := $'({expr1.ExprStr}+{expr2.ExprStr})';
                complexity += 1;
              end;
            '-':
              begin
                res := expr1.Value - expr2.Value;
                exprStr := $'({expr1.ExprStr}-{expr2.ExprStr})';
                complexity += 1;
              end;
            '*':
              begin
                res := expr1.Value * expr2.Value;
                exprStr := $'({expr1.ExprStr}*{expr2.ExprStr})';
                complexity += 2;
              end;
            '/':
              begin
                if Abs(expr2.Value) < 1e-12 then Exit(nil);
                res := expr1.Value / expr2.Value;
                exprStr := $'({expr1.ExprStr}/{expr2.ExprStr})';
                complexity += 3;
              end;
          end;
          
          // Проверка на переполнение
          if (Abs(res) > 1e100) or ((Abs(res) < 1e-100) and (res <> 0)) then
            Exit(nil);
          
          if real.IsNaN(res) then Exit(nil);
          
          Result := new Expression(res, exprStr, totalFours, complexity);
          
        except
          on e: Exception do
            Exit(nil);
        end;
      end;
      
      function GenerateBaseExpressions: List<Expression>;
      begin
        Result := new List<Expression>;
        
        // Сама цифра 4 (1 четвёрка)
        Result.Add(new Expression(4.0, '4', 1, 0));
        
        // Десятичная точка .4
        if AllowedUnary.Contains('.') then
          Result.Add(new Expression(0.4, '.4', 1, 1));
        
        // Периодическая дробь .4~ = 4/9
        if AllowedUnary.Contains('~') then
          Result.Add(new Expression(4.0/9.0, '.4~', 1, 2));
        
        // sqrt(4) = 2
        if AllowedUnary.Contains('sqrt') then
          Result.Add(new Expression(2.0, 'sqrt(4)', 1, 5));
        
        // 4! = 24
        if AllowedUnary.Contains('!') then
          Result.Add(new Expression(24.0, '4!', 1, 10));
        
        // 4^2 = 16
        if AllowedUnary.Contains('^') then
          Result.Add(new Expression(16.0, '4^2', 1, 3));
      end;
      
      function Solve(opsSet: string): Dictionary<integer, Expression>;
      begin
        ConfigureOperations(opsSet);
        
        // Храним выражения по количеству четвёрок
        var expressionsByFours := new Dictionary<integer, HashSet<Expression>>;
        for var i := 1 to RequiredFours do
          expressionsByFours[i] := new HashSet<Expression>;
        
        // Шаг 1: Базовые выражения
        var baseExprs := GenerateBaseExpressions;
        foreach var expr in baseExprs do
          if expr.UsedFours <= RequiredFours then
            expressionsByFours[expr.UsedFours].Add(expr);
        
        // Шаг 2: Комбинируем выражения
        for var totalFours := 2 to RequiredFours do
        begin
          // Комбинируем выражения разного размера
          for var leftFours := 1 to totalFours-1 do
          begin
            var rightFours := totalFours - leftFours;
            
            var leftExpressions := expressionsByFours[leftFours].ToList;
            var rightExpressions := expressionsByFours[rightFours].ToList;
            
            foreach var expr1 in leftExpressions do
              foreach var expr2 in rightExpressions do
                foreach var op in AllowedOps do
                begin
                  var newExpr := ApplyBinaryOperation(expr1, expr2, op);
                  if (newExpr <> nil) and newExpr.IsValid then
                  begin
                    expressionsByFours[totalFours].Add(newExpr);
                    Stats.ExpressionsGenerated += 1;
                  end;
                end;
          end;
          
          // Применяем унарные операции
          var currentExprs := expressionsByFours[totalFours].ToList;
          foreach var expr in currentExprs do
            foreach var unaryOp in AllowedUnary do
            begin
              var newExpr := ApplyUnaryOperation(expr, unaryOp);
              if (newExpr <> nil) and newExpr.IsValid then
                expressionsByFours[totalFours].Add(newExpr);
            end;
        end;
        
        // Шаг 3: СОБИРАЕМ ВЫРАЖЕНИЯ С РОВНО 4 ЧЕТВЁРКАМИ
        AllExpressionsExact := expressionsByFours[RequiredFours].ToList;
        Stats.ExpressionsWith4Fours := AllExpressionsExact.Count;
        
        Writeln('Найдено ', Stats.ExpressionsWith4Fours, ' выражений с РОВНО ', RequiredFours, ' четвёрками');
        
        // Шаг 4: Собираем результаты для целых чисел
        var bestExpressions := new Dictionary<integer, Expression>;
        
        foreach var expr in AllExpressionsExact do
        begin
          if not expr.IsValid then continue;
          
          for var target := 0 to MaxInteger do
          begin
            if Abs(expr.Value - target) < 1e-10 then
            begin
              if not bestExpressions.ContainsKey(target) then
                bestExpressions[target] := expr
              else if (expr.Complexity < bestExpressions[target].Complexity) or
                      ((expr.Complexity = bestExpressions[target].Complexity) and
                       (expr.ExprStr.Length < bestExpressions[target].ExprStr.Length)) then
                bestExpressions[target] := expr;
              break;
            end;
          end;
        end;
        
        // Формируем окончательные результаты
        ResultsExactFours.Clear;
        for var num := 0 to MaxInteger do
          if bestExpressions.ContainsKey(num) then
            ResultsExactFours[num] := bestExpressions[num];
        
        Stats.NumbersFound := ResultsExactFours.Count;
        Writeln('Из них дают целые числа: ', Stats.NumbersFound, ' выражений');
        
        Result := ResultsExactFours;
      end;
      
      procedure PrintResults(results: Dictionary<integer, Expression>; showGaps: boolean := true);
      begin
        if results.Count = 0 then
        begin
          Writeln('Решений с 4 четвёрками не найдено.');
          Exit;
        end;
        
        // Находим непрерывный ряд
        var continuousUntil := 0;
        for var i := 0 to MaxInteger do
          if results.ContainsKey(i) then
            continuousUntil := i
          else
            break;
                
        Writeln;        
        for var num := 0 to 200 do
        begin
          if results.ContainsKey(num) then
          begin
            var expr := results[num];
            var exprFormatted := expr.ExprStr.Replace('*', '×').Replace('/', '÷');
            Writeln($'  {num,2} = {exprFormatted}');
          end
          else if num <= continuousUntil then
            Writeln($'  {num,2} = ??? (должно быть найдено!)')
          else
            Writeln($'  {num,2} = не найдено');
        end;
      end;
  end;


procedure InteractiveDemo;
begin 
  var solver := new FourFoursSolver(200);
  
  Writeln;
  Writeln('1. Базовый: только + - × ÷');
  Writeln('2. Средний: + - × ÷, факториал, корень, точка');
  Writeln('3. Продвинутый: + - × ÷, факториал, корень, точка, floor/ceil');
  
  while true do
  begin
    Write(#10 + 'Выберите набор операций (1-3): ');
    var choice := ReadString.Trim;
    
    if (choice.Length = 1) and (choice[1] in ['1'..'3']) then begin
      var opsSet := '';
      case choice[1] of
        '1': opsSet := 'basic';
        '2': opsSet := 'intermediate';
        '3': opsSet := 'advanced';
        '4': opsSet := 'extreme';
      end;
      
      var results := solver.Solve(opsSet);
      solver.PrintResults(results, true);
    end
    else
      Writeln('Пожалуйста, выберите от 1 до 3');
  end;
end;

begin
   WriteLn(' Четыре четвёрки');
   WriteLn;  
   InteractiveDemo;
end.